//-----------------------------------------------------------------------------
// Copyright:      RAD Electronic Co. LTD,
// Author:         jaruwit supa, Base on FontEditor written by H. Reddmann
//								 Modified by Sh. Nourbakhsh Rad for Persian and Arabic font
//								 and fixed some errors
// Remarks:        
// known Problems: none
// Version:        20.03.2011
// Description:    Font Library
//-----------------------------------------------------------------------------

#include "Font.h"
#include "PE_Map.h"


uint8_t 						*FontPointer; 								// Font	Pointer
uint8_t 						rot 						= 0;					// Rot 0=0, 1=90
uint8_t 						Reverse					= 0;					// Reverse text direction on the display
uint8_t							Horizontal			=	1;					// 1: Portrait - 240x320 ,   2: Landscape - 320x240

uint16_t 						FgColor 				= BLACK;			// Text fg color, 5-6-5 RGB
uint16_t 						BkColor 				= WHITE;			// Text bk color, 5-6-5 RGB
uint8_t 						FontFixed				= 0;					// Text type 0=Proportional , 1=Fixed
uint8_t							NonTransparence = 0;					// Transparent 0=No, 1=Yes

uint16_t 						fontSize;											// size of current font
uint8_t 						firstchar;										// first character noumber of current font
uint8_t 						lastchar;											// last character noumber of current font
uint8_t 						charwidth;										// current character width register

uint8_t							FontWidth;										// max width of font
uint8_t							FontHeight;										// max height of font
uint8_t 						FontXScale 			= 1;					// X size of font
uint8_t 						FontYScale 			= 1;					// Y size of font
uint8_t 						FontSpace 			= 1;					// space between char

unsigned int 				cursorX 				= 0;					// x position
unsigned int 				cursorY 				= 0;					// y position

uint8_t							peLETTER				= E_LETTER;		// English or Persian letter

uint8_t 						prevLet 				= 0xFF;				// previous persian character register
uint8_t	 						nextLet 				= 0xFF; 			// next persian character register

#define FONT_HEADER_SIZE            7     				// header size of fonts

//********************************************
//********************************************

//**** select font
void LcdFont(uint8_t *pointer)
{
	FontPointer 			= pointer;
	FontWidth 				= pgm_read_byte_near(&FontPointer[2]);
	FontHeight				= pgm_read_byte_near(&FontPointer[3]);

	fontSize					=	pgm_read_word_near(&FontPointer[0]);
	firstchar 				=	pgm_read_byte_near(&FontPointer[5]);
	lastchar 					=	pgm_read_byte_near(&FontPointer[6]);
}	//LcdFont

//**** draw char
void PutChar(unsigned char c)
{
  if(peLETTER == P_LETTER)
		PutCharPE(c);
	else
			PutCharEN(c);
}	//PutChar

void SetLine(unsigned char line,unsigned char column)
{
	if(peLETTER == P_LETTER)
		cursorX =GetMaxX() - (unsigned int)FontWidth  * FontXScale * column;
	else
		cursorX =(unsigned int)FontWidth  * FontXScale * column;

	cursorY =(unsigned int)FontHeight * FontYScale * line;
}	//SetCursor

unsigned int CalcTextWidth(char *Text)
{
	if(peLETTER == E_LETTER)
		return CalcTextWidthEN(Text);
	else
		return CalcTextWidthPE(Text);
}	//CalcTextWidth

unsigned int CalcTextHeight(char *Text)
{
	return FontHeight * FontYScale;
}	//CalcTextHeight

//**** draw string
void Puts(char *Text)
{	
	unsigned char 	c;
	
  if(peLETTER == P_LETTER)
		PutsPE(Text);
	else
	{
		while((c = *Text++))
			PutCharEN(c);
	}
}	//Puts

void PutsP(char *Text)
{
	unsigned char 	c;
	
	while((c = pgm_read_byte(Text++)))
	{
		PutChar(c);
	}
}	//PutsP

void PutLong(uint32_t src, uint8_t digit, uint8_t decimal, uint8_t Parameter)
{
	char 			text[10+1];
	uint8_t 	i;
	Lt_Mode		cMode;
	
	cMode = peLETTER;
	SetLetter(E_LETTER);
	
	text[digit] = 0;
	for(i = digit; i >0; i--)
	{
		text[i-1] =(src % 10) + 48;
		src = src/10;
	}//for digit
	
	if (Parameter & (STYLE_NO_ZERO | STYLE_NO_SPACE))
	{
		while(text[i] == '0')
		{
		
			if (Parameter & (STYLE_NO_ZERO))
				PutChar(' ');
			
			i++;
			if (i > digit - decimal -1 -1) // include lead 0.xx
				break;
		}
	}
	
	while(text[i])
	{
		if (i == digit - decimal) // include lead 0.xx
			PutChar('.');
		
		PutChar(text[i]);
		i++;
	}

	SetLetter(cMode);
}	//PutLong

void PutInt(uint16_t src, uint8_t digit, uint8_t decimal, uint8_t Parameter)
{
	unsigned long 		n = src;
	
	PutLong(n, digit, decimal, Parameter);
}	//PutInt

void PutByte(uint8_t src, uint8_t digit, uint8_t decimal, uint8_t Parameter)
{
	unsigned long 		n = src;
	
	PutLong(n, digit, decimal, Parameter);
}	//PutByte

void PutText(uint16_t left, uint16_t top, uint16_t right, uint16_t bottom, char *str, uint8_t style)
{
	uint16_t 			sizeofstring;

	sizeofstring = CalcTextWidth(str);
	top = top + (bottom - top - (FontHeight*FontYScale))/2;

	
	if(peLETTER == E_LETTER)
	{
		switch (style & ALINE_MARK)
		{
			case ALINE_LEFT:
				left = left + FontSpace;
				break;
	
			case ALINE_CENTER:
					left = left + (right - left - sizeofstring) / 2;
				break;
	
			case ALINE_RIGHT:
				if (right > sizeofstring)
					left = right - sizeofstring;
				break;
		}//switch

		SetCursorX(left);
	}
	else
	{
		switch (style & ALINE_MARK)
		{
			case ALINE_RIGHT:
				break;
	
			case ALINE_CENTER:
					right = right - (right - left - sizeofstring) / 2;
				break;
	
			case ALINE_LEFT:
				right = left + sizeofstring;
				right = right + FontSpace;
				break;
		}//switch

		SetCursorX(right);		
	}
	
	SetCursorY(top);
	Puts(str);
}	//PutText

void PutTextP(uint16_t left, uint16_t top, uint16_t right, uint16_t bottom, const char *str, uint8_t style)
{
	char 				strP[100+1];
	
	strncpy_P(strP, str, strnlen_P(str,100));
	strP[100] = 0;	// terminate string
	PutText(left, top, right, bottom, strP, style);
}	//PutTextP

void TextBox(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, char *str, uint8_t style)
{
	unsigned int 			r;

	if (style & BORDER_RECT)
	{
		r = (style & BORDER_BEVEL) >> 4;	// get bevel radius
		if (r)	// bevel
		{
			if (style & BORDER_FILL)
				BevelFill(x1+1, y1+1, x2-1, y2-1, r, BkColor);
			
			Bevel(x1, y1, x2, y2, r, FgColor);
		}
		else	// rect
		{
			if (style & BORDER_FILL)
				RectangleFill(x1+1, y1+1, x2-1, y2-1, BkColor);

			Rectangle(x1, y1, x2, y2, FgColor);
		}
	}
	else if (style & BORDER_FILL)
	{
		RectangleFill(x1, y1, x2, y2, BkColor);
	}

	if ((style & BORDER_RECT) || (style & BORDER_FILL))
	    PutText(x1+1, y1+1, x2-1, y2-1, str, style & ALINE_MARK);
    else
	    PutText(x1, y1, x2, y2, str, style & ALINE_MARK);
}	//TextBox

void TextBoxP(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, char *str, uint8_t style)
{
	char 		strP[100+1];
	
	strncpy_P(strP, str, strnlen_P(str,100));
	strP[100] = 0;	// terminate string
	TextBox(x1, y1, x2, y2, strP, style);
}	//TextBoxP

/////////////////////////////////////
void SetLetter(Lt_Mode L)
{
	peLETTER = L;
	
	if(L == P_LETTER)
		cursorX = GetMaxX();
	else
		cursorX = 0;
}	//SetLetter

void PutsPE(char *str)
{	
	unsigned char 	curnLet,	dumyLet;
	unsigned char 	Pstat, stat, Nstat;
	
	unsigned char		backFontSpace = FontSpace;
	char 						strTemp[50];
	unsigned char 	i = 0, j = 0;

 	while((curnLet = *str++))
	{ 		
 		switch(curnLet)
 		{	
 			case 0x81:				//peh
    		curnLet = 0xEA;
 				break;
 			case 0x8D:				//cheh
    		curnLet = 0xEB;
 				break;
 			case 0x8E:				//zheh
    		curnLet = 0xEE;
 				break;
 			case 0x90:				//geh
    		curnLet = 0xEF;
 				break;
 			case 0xE1:				//laa
    		nextLet = *str++;
    		
    		if(nextLet == 0xC7)
    			curnLet = 0xFB;
    		else
    		{
    			dumyLet = *str--;
    			curnLet = 0xE1;
    		}
 				break;
 			case 0x3F:				//persian question mark
    		curnLet = 0xBF;
 				break;
 			case 0x3B:				//persian semicolon
    		curnLet = 0x1F;
 				break;
 			case 0x2C:				//persian comma
    		curnLet = 0x1D;
 				break;
 			case 0x2E:				//persian point
    		curnLet = 0x1C;
 				break; 				
		}//switch curnLet
		
		//--------------------------------
  	nextLet = *str++;
  	dumyLet	= *str--;
 	
 		switch(nextLet)
 		{	
 			case 0x81:				//peh
    		nextLet = 0xEA;
 				break;
 			case 0x8D:				//cheh
    		nextLet = 0xEB;
 				break;
 			case 0x8E:				//zheh
    		nextLet = 0xEE;
 				break;
 			case 0x90:				//geh
    		nextLet = 0xEF;
 				break;
 			case 0x3F:				//persian question mark
    		nextLet = 0xBF;
 				break;
 			case 0x3B:				//persian semicolon
    		nextLet = 0x1F;
 				break;
 			case 0x2C:				//persian comma
    		nextLet = 0x1D;
 				break;
 			case 0x2E:				//persian point
    		nextLet = 0x1C;
 				break; 			
		}//switch nextLet

		if(curnLet > 0xC0)
		{		
			if(prevLet > 0xC0)
				Pstat = (pgm_read_byte(&(PEmap[prevLet-0xC1][5])));		// 1: prevLet attach to curnLet -- 0: prevLet don't attach to curnLet
			else
				Pstat = 0;
				
			if(nextLet > 0xC0)
				Nstat = (pgm_read_byte(&(PEmap[nextLet-0xC1][4])));		// 1: nextLet attach to curnLet -- 0: nextLet don't attach to curnLet
			else
				Nstat = 0;
			
			//			Pstat	|	Nstat	|	stat
			//		 -------+-------+------
			//				0		|		0		|		0				curnLet, don't attach to prevLet and nextLet
			//				0		|		1		|		1				curnLet, don't attach to prevLet and attach to nextLet
			//				1		|		0		|		2				curnLet, attach to prevLet and don't attach to nextLet
			//				1		|		1		|		3				curnLet, attach to prevLet and nextLet
			
			stat = (Pstat<<1) | Nstat;		
			
			if(stat>1)	FontSpace = 0;
			else 				FontSpace = backFontSpace;
			
			PutCharPE(pgm_read_byte(&(PEmap[curnLet-0xC1][stat])));
			FontSpace = backFontSpace;
		}
		else	//Original 'curnLet' below 193 without (     )  -- for digits and symbols!
		{
  		if((curnLet >= '0') && (curnLet <= '9'))		// 0 to 9
  		{
  			i++;
  			strTemp[i] 	 = curnLet;

  			if(!((nextLet >= '0') && (nextLet <= '9')))
				{
					for(j=i; j>0; j--)
						PutCharPE(strTemp[j]-0x20);

					i = 0;
				}
			}		
			else
				PutCharPE(curnLet);
		}
		
		prevLet = curnLet;
	}//while

 	prevLet = 0xFF;
}	//PutsPE

void PutCharPE(unsigned char c)
{
	uint8_t 			byte 					= 0;
	uint8_t 			bitoffset 		= 0;
	uint8_t 			maske 				= 0;
	uint16_t 			bcounter 			= 0;
	uint16_t 			bitsbischar 	= 0;
	uint16_t 			bytesbischar	= 0;
	uint16_t 			xPos,yPos;

	// get current character width
	charwidth	= pgm_read_byte_near(&FontPointer[7+c-firstchar]);
	
	// line feed, goto next line
	if(c == '\n')
	{
		cursorX = GetMaxX();
		cursorY = cursorY + (unsigned int)FontHeight * FontYScale; 
		
		return;
	}

	// charactor out of range.
  if( (c < firstchar) || (c > lastchar) || (fontSize == 0)) 
  	return;

	// charactor is not in list.
	if (charwidth == 0)
		return;

	// sara thai font. line remain at last position
	if(FontFixed)
		charwidth = FontWidth;
	
	// line adjust
	if(((int)cursorX - charwidth * FontXScale) < 0)
  {
		cursorY = cursorY   + (unsigned int)FontHeight * FontYScale; 
		cursorX = GetMaxX() - charwidth * FontXScale;
	}

	// adjust cursor to current position - persian
	cursorX	-= charwidth * FontXScale + FontSpace;

	// calculate current character position on the table
	for(uint8_t Ccounter = 0; Ccounter < c-firstchar; Ccounter++)
		bitsbischar += (pgm_read_byte_near(&FontPointer[Ccounter+7]));	//c0_width +...+ cn_width


	bitsbischar 		*= FontHeight;										//c_widths * FH
	bitsbischar 		+= ((lastchar-firstchar)+7)*8;		//plus font headers
	bytesbischar 		 = bitsbischar/8;									//
	bitoffset 			 = bitsbischar % 8;								//
	maske						 = bitoffset % 8;									//

	// draw character
	for(uint8_t xc = 0; xc < charwidth; xc++)
	{
		for(uint8_t yc = 0; yc < FontHeight; yc++)
		{
			if(maske > 7)
			{
				maske = 0;
				bcounter+=1;
			};
			byte = pgm_read_byte_near(&FontPointer[bytesbischar + bcounter]+1);

			xPos = (uint16_t)xc*FontXScale + cursorX;
			
			if(Horizontal)	//128x64
			{
				if(Reverse)
					yPos = (((uint16_t)FontHeight-yc)*FontYScale + cursorY);
				else
					yPos = (((uint16_t)yc)*FontYScale + cursorY);     
			}
			else						//64x128
			{
				if(Reverse)
					yPos = (((uint16_t)FontHeight-yc)*FontYScale + cursorY);
				else
					yPos = (((uint16_t)yc)*FontYScale + cursorY);
				
			}	//Horizontal

			for(uint8_t sx = 0; sx < FontXScale; sx++)
			{
				for(uint8_t sy = 0; sy < FontYScale; sy++)
				{
					if(bit_is_set(byte,maske))
					{
						if(rot)		PutPixel(yPos +sy,xPos+sx,FgColor);
						else			PutPixel(xPos +sx,yPos+sy,FgColor);
					}
					else
					{
						if(NonTransparence)
						{
							if(rot)		PutPixel(yPos +sy,xPos+sx,BkColor);
							else			PutPixel(xPos +sx,yPos+sy,BkColor);
						}
					}
				}//for sy
			}//for sx
			
			maske++;
		}//for yc
	}//for xc	
}	//PutCharPE

void PutCharEN(unsigned char c)
{
	uint8_t 			byte 					= 0;
	uint8_t 			bitoffset 		= 0;
	uint8_t 			maske 				= 0;
	uint16_t 			bcounter 			= 0;
	uint16_t 			bitsbischar 	= 0;
	uint16_t 			bytesbischar	= 0;
	uint16_t 			xPos,yPos;

	// get current character width
	charwidth	= pgm_read_byte_near(&FontPointer[7+c-firstchar]);
	
	// line feed, goto next line
	if(c == '\n')
	{
		cursorX = 0;
		cursorY = cursorY + (unsigned int)FontHeight * FontYScale; 
		
		return;
	}

	// charactor out of range.
  if( (c < firstchar) || (c > lastchar) || (fontSize == 0)) 
  	return;

	// charactor is not in list.
	if (charwidth == 0)
		return;

	// sara thai font. line remain at last position
	if(FontFixed)
		charwidth = FontWidth;
	else
	{
		//english spesial fonts!
		if (((c >= 0xd4) && (c <= 0xda)) || 
	  	  ((c >= 0xe7) && (c <= 0xec)) ||
				 (c == 0xd1))
		{
			cursorX = cursorX - charwidth * FontXScale;
		}
	}

	// fixed width for digit
 	if((c >= '0') && (c <= '9'))		// english : 0 to 9
		charwidth = pgm_read_byte_near(&FontPointer[7+'0'-firstchar]);		//width reference = ZERO
	
	// line adjust
	if((cursorX + charwidth * FontXScale) > GetMaxX())
	{
		cursorY = cursorY + (unsigned int)FontHeight * FontYScale; 
		cursorX = 0;
	}

	// calculate current character position on the table
	for(uint8_t Ccounter = 0; Ccounter < c-firstchar; Ccounter++)
		bitsbischar += (pgm_read_byte_near(&FontPointer[Ccounter+7]));	//c0_width +...+ cn_width


	bitsbischar 		*= FontHeight;										//c_widths * FH
	bitsbischar 		+= ((lastchar-firstchar)+7)*8;		//plus font headers
	bytesbischar 		 = bitsbischar/8;									//
	bitoffset 			 = bitsbischar % 8;								//
	maske						 = bitoffset % 8;									//

	// draw character
	for(uint8_t xc = 0; xc < charwidth; xc++)
	{
		for(uint8_t yc = 0; yc < FontHeight; yc++)
		{
			if(maske > 7)
			{
				maske = 0;
				bcounter+=1;
			};
			byte = pgm_read_byte_near(&FontPointer[bytesbischar + bcounter]+1);

			xPos = (uint16_t)xc*FontXScale + cursorX;
			
			if(Horizontal)	//128x64
			{
				if(Reverse)
					yPos = (((uint16_t)FontHeight-yc)*FontYScale + cursorY);
				else
					yPos = (((uint16_t)yc)*FontYScale + cursorY);     
			}
			else						//64x128
			{
				if(Reverse)
					yPos = (((uint16_t)FontHeight-yc)*FontYScale + cursorY);
				else
					yPos = (((uint16_t)yc)*FontYScale + cursorY);
				
			}	//Horizontal

			for(uint8_t sx = 0; sx < FontXScale; sx++)
			{
				for(uint8_t sy = 0; sy < FontYScale; sy++)
				{
					if(bit_is_set(byte,maske))
					{
						if(rot)		PutPixel(yPos +sy,xPos+sx,FgColor);
						else			PutPixel(xPos +sx,yPos+sy,FgColor);
					}
					else
					{
						if(NonTransparence)
						{
							if(rot)		PutPixel(yPos +sy,xPos+sx,BkColor);
							else			PutPixel(xPos +sx,yPos+sy,BkColor);
						}
					}
				}//for sy
			}//for sx
			
			maske++;
		}//for yc
	}//for xc
	
	// adjust cursor to next position - english
	cursorX	+= charwidth * FontXScale + FontSpace;
}	//PutCharEN

unsigned int CalcTextWidthEN(char *str)
{
	unsigned int 		strSize = 0;
	unsigned char 	c;

	while((c = *str++))
	{
		if(c == '\n')			continue;
		
		if(FontFixed)
			strSize += FontWidth * FontXScale;
		else
		{
		  if((c < firstchar) || (c > lastchar)) 
				charwidth = FontWidth;
			else
			{
				charwidth = pgm_read_byte_near(&FontPointer[7+c-firstchar]);

				//english spesial fonts!
				if (((c >= 0xd4) && (c <= 0xda)) || 
		    		((c >= 0xe7) && (c <= 0xec)) ||
				  	 (c == 0xd1))

					charwidth = 0;
			}
			
			strSize += charwidth * FontXScale;
		}
		
		strSize += FontSpace;
	}//while
	
	return strSize ;
}	//CalcTextWidthEN

unsigned int CalcTextWidthPE(char *str)
{
	unsigned char 	curnLet,	dumyLet;
	unsigned char 	Pstat, stat, Nstat;
	
	unsigned char		backFontSpace = FontSpace;
	unsigned int 		strSize = 0;
	
 	while((curnLet = *str++))
	{ 		
		if(curnLet == '\n')			continue;
	
		if(FontFixed)
			strSize += FontWidth * FontXScale;
		else
		{
		 	if((curnLet < firstchar) || (curnLet > lastchar)) 
				charwidth = FontWidth;
			else
			{
		 		switch(curnLet)
		 		{	
		 			case 0x81:				//peh
		    		curnLet = 0xEA;
		 				break;
		 			case 0x8D:				//cheh
		    		curnLet = 0xEB;
		 				break;
		 			case 0x8E:				//zheh
		    		curnLet = 0xEE;
		 				break;
		 			case 0x90:				//geh
		    		curnLet = 0xEF;
		 				break;
		 			case 0xE1:				//laa
		    		nextLet = *str++;
		    		
		    		if(nextLet == 0xC7)
		    			curnLet = 0xFB;
		    		else
		    		{
		    			dumyLet = *str--;
		    			curnLet = 0xE1;
		    		}
		 				break;
		 			case 0x30:				//persian digits 0...9
					case 0x31:
					case 0x32:
					case 0x33:
					case 0x34:
					case 0x35:
					case 0x36:
					case 0x37:
					case 0x38:
					case 0x39:
						curnLet -= 0x20;
		 				break;
		 			case 0x3F:				//persian question mark
		    		curnLet = 0xBF;
		 				break;
		 			case 0x3B:				//persian semicolon
		    		curnLet = 0x1F;
		 				break;
		 			case 0x2C:				//persian comma
		    		curnLet = 0x1D;
		 				break;
		 			case 0x2E:				//persian point
		    		curnLet = 0x1C;
		 				break; 				
				}//switch curnLet
		
				//--------------------------------
		  	nextLet = *str++;
		  	dumyLet	= *str--;
		  	
		 		switch(nextLet)
		 		{	
		 			case 0x81:				//peh
		    		nextLet = 0xEA;
		 				break;
		 			case 0x8D:				//cheh
		    		nextLet = 0xEB;
		 				break;
		 			case 0x8E:				//zheh
		    		nextLet = 0xEE;
		 				break;
		 			case 0x90:				//geh
		    		nextLet = 0xEF;
		 				break;
		 			case 0x30:				//0...9
					case 0x31:
					case 0x32:
					case 0x33:
					case 0x34:
					case 0x35:
					case 0x36:
					case 0x37:
					case 0x38:
					case 0x39:
						nextLet -= 0x20;
		 				break; 			
		 			case 0x3F:				//persian question mark
		    		nextLet = 0xBF;
		 				break;
		 			case 0x3B:				//persian semicolon
		    		nextLet = 0x1F;
		 				break;
		 			case 0x2C:				//persian comma
		    		nextLet = 0x1D;
		 				break;
		 			case 0x2E:				//persian point
		    		nextLet = 0x1C;
		 				break; 				
				}//switch nextLet

				if(curnLet > 0xC0)
				{		
					if(prevLet > 0xC0)
						Pstat = (pgm_read_byte(&(PEmap[prevLet-0xC1][5])));
					else
						Pstat = 0;
						
					if(nextLet > 0xC0)
						Nstat = (pgm_read_byte(&(PEmap[nextLet-0xC1][4])));
					else
						Nstat = 0;
					
					stat = (Pstat<<1) | Nstat;		
					
					if(stat>1)	FontSpace = 0;
					else 				FontSpace = backFontSpace;
					////////////////
					dumyLet = pgm_read_byte(&(PEmap[curnLet-0xC1][stat]));
					
					charwidth = pgm_read_byte_near(&FontPointer[7+dumyLet-firstchar]);
					strSize += charwidth * FontXScale;
					strSize += FontSpace;
					////////////////
					FontSpace = backFontSpace;
				}
				else
				{
					charwidth = pgm_read_byte_near(&FontPointer[7+curnLet-firstchar]);
					strSize += charwidth * FontXScale;
					strSize += FontSpace;
				}
				
				prevLet = curnLet;
			}
		}
	}//while

 	prevLet = 0xFF;
	return strSize ;
}	//CalcTextWidthPE
